<?php

namespace Drupal\commerce_shipping\Access;

use Drupal\Core\Access\AccessResult;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Routing\Access\AccessInterface;
use Drupal\Core\Routing\RouteMatchInterface;
use Drupal\Core\Session\AccountInterface;
use Symfony\Component\Routing\Route;

/**
 * Defines an access checker for the Shipment collection route.
 */
class ShipmentCollectionAccessCheck implements AccessInterface {

  /**
   * Constructs a new ShipmentCollectionAccessCheck object.
   *
   * @param \Drupal\Core\Entity\EntityTypeManagerInterface $entityTypeManager
   *   The entity type manager.
   */
  public function __construct(protected EntityTypeManagerInterface $entityTypeManager) {}

  /**
   * Checks access to the Shipment collection.
   *
   * @param \Symfony\Component\Routing\Route $route
   *   The route to check against.
   * @param \Drupal\Core\Routing\RouteMatchInterface $route_match
   *   The route match.
   * @param \Drupal\Core\Session\AccountInterface $account
   *   The currently logged in account.
   *
   * @return \Drupal\Core\Access\AccessResultInterface
   *   The access result.
   */
  public function access(Route $route, RouteMatchInterface $route_match, AccountInterface $account) {
    /** @var \Drupal\commerce_order\Entity\OrderInterface $order */
    $order = $route_match->getParameter('commerce_order');
    $order_type_storage = $this->entityTypeManager->getStorage('commerce_order_type');
    /** @var \Drupal\commerce_order\Entity\OrderTypeInterface $order_type */
    $order_type = $order_type_storage->load($order->bundle());
    $shipment_type_id = $order_type->getThirdPartySetting('commerce_shipping', 'shipment_type');
    // Check if this is a cart order.
    $order_is_cart = $order->hasField('cart') && $order->get('cart')->value;
    $access_control_handler = $this->entityTypeManager->getAccessControlHandler('commerce_shipment');

    // Only allow access if order type has a corresponding shipment type.
    if (empty($shipment_type_id)) {
      return AccessResult::forbidden()
        ->addCacheableDependency($order_type);
    }

    return AccessResult::allowedIf($access_control_handler->createAccess($shipment_type_id, $account))
      ->andIf(AccessResult::allowedIf(!$order_is_cart))
      ->addCacheableDependency($order_type)
      ->addCacheableDependency($order);
  }

}
